<?php
namespace App\Controller\Component;

use \DateTime;
use \DateTimeZone;
use Cake\Controller\Component;
use Cake\Utility\Inflector;
use Cake\Core\Configure;

use Google\AdsApi\Common\OAuth2TokenBuilder;


use Google\AdsApi\AdManager\AdManagerSession;
use Google\AdsApi\AdManager\AdManagerServices;
use Google\AdsApi\AdManager\AdManagerSessionBuilder;
use Google\AdsApi\AdManager\Util\v202211\StatementBuilder;
use Google\AdsApi\AdManager\v202211\ServiceFactory;
use Google\AdsApi\AdManager\v202211\Dimension;
use Google\AdsApi\AdManager\v202211\DimensionAttribute;
use Google\AdsApi\AdManager\v202211\Column;
use Google\AdsApi\AdManager\v202211\DateRangeType;
use Google\AdsApi\AdManager\v202211\ReportJob;
use Google\AdsApi\AdManager\v202211\ReportQuery;
use Google\AdsApi\AdManager\v202211\ReportService;
use Google\AdsApi\AdManager\v202211\ExportFormat;
use Google\AdsApi\AdManager\v202211\Company;
use Google\AdsApi\AdManager\v202211\CompanyType;
use Google\AdsApi\AdManager\v202211\ReportDownloadOptions;
use Google\AdsApi\AdManager\v202211\Order;
use Google\AdsApi\AdManager\v202211\StartDateTimeType;
use Google\AdsApi\AdManager\v202211\ApproveOrders as ApproveOrdersAction;
use Google\AdsApi\AdManager\v202211\CustomTargetingKeyType;
use Google\AdsApi\AdManager\Util\v202211\ReportDownloader;
use Google\AdsApi\AdManager\Util\v202211\AdManagerDateTimes;

//use Google\AdsApi\Common\OAuth2TokenBuilder;

class AdManager2023Component extends Component {

	var $connected = false;
	var $credential;
	var $session;
	var $primary_trafficker_id = 227598299;

	public function initialize(array $config): void 
	{
		$this->controller = $this->_registry->getController();
	}
	
	function connect(){
		if($this->connected){
			return true;
		}
		
		require_once 'vendor/googleads/googleads-php-lib-jan-17-2023/vendor/autoload.php';

		
		$this->credential = (new OAuth2TokenBuilder())
			->fromFile('credentials/google_ads_php.ini')
			// ->fromFile(CONFIG . 'google_ads_php.ini')
			->build();

		// $this->session = (new GoogleAdsClientBuilder())
		// 	->fromFile('../credentials/google_ads_php.ini')
		// 	->withOAuth2Credential($this->credential)
		// 	->build()
		// ;
		
		$this->session = (new AdManagerSessionBuilder())
		// ->fromFile('credentials/google_ads_php.ini')
		->fromFile(CONFIG . 'google_ads_php.ini')
		->withOAuth2Credential($this->credential)
		->build();
		
		return true;
	}
	
	function getCompanies(){
		$this->connect();

		$serviceFactory = new ServiceFactory();
		$companyService = $serviceFactory->createCompanyService($this->session);
		
		// Create a statement to select creatives.
		$pageSize = StatementBuilder::SUGGESTED_PAGE_LIMIT;
		$statementBuilder = (new StatementBuilder())
			->orderBy('name ASC')
			->limit($pageSize)
		;
		// Retrieve a small amount of creatives at a time, paging
		// through until all creatives have been retrieved.
		
		$companies = [];
		
		$totalResultSetSize = 0;
		do {
			$page = $companyService->getCompaniesByStatement(
				$statementBuilder->toStatement()
			);
			// Print out some information for each creative.
			if ($page->getResults() !== null) {
				$totalResultSetSize = $page->getTotalResultSetSize();
				$i = $page->getStartIndex();
				foreach ($page->getResults() as $company) {
					$companies[] = [
						'id' => $company->getId(),
						'name' => $company->getName()
					];
				}
			}
			$statementBuilder->increaseOffsetBy($pageSize);
		} while ($statementBuilder->getOffset() < $totalResultSetSize);
		
		
		return $companies;
	}
	
	function reportQuery($dimensions, $attributes, $columns, $startDate, $endDate, $conditions=[]){
		$this->connect();

		$reportQuery = new ReportQuery();
		$reportQuery->setDimensions($dimensions);
		$reportQuery->setDimensionAttributes($attributes);
		$reportQuery->setColumns($columns);

		// Create statement to filter for an order.
		$statementBuilder = (new StatementBuilder());
		foreach($conditions as $key=>$value){
			$statementBuilder = $statementBuilder->where($key.' = :'.$key)->withBindVariableValue($key,$value);
		}
		
		
		// Set the filter statement.
		$reportQuery->setStatement($statementBuilder->toStatement());
		
		// Set the start and end dates or choose a dynamic date range type.
		$reportQuery->setDateRangeType(DateRangeType::CUSTOM_DATE);
		$reportQuery->setStartDate(
			AdManagerDateTimes::fromDateTime(
				new DateTime(
					$startDate,
					new DateTimeZone('America/Vancouver')
				)
			)
				->getDate()
		);
		$reportQuery->setEndDate(
			AdManagerDateTimes::fromDateTime(
				new DateTime(
					$endDate,
					new DateTimeZone('America/Vancouver')
				)
			)
				->getDate()
		);
		
		return $reportQuery;
	}
	
	function reportJob($reportQuery){
		$this->connect();

		
		$reportJob = new ReportJob();
		$reportJob->setReportQuery($reportQuery);
		
		$adManagerServices =  new AdManagerServices();
		$reportService = $adManagerServices->get(
			$this->session,
			ReportService::class
		);


		// exit();
		$reportJob = $reportService->runReportJob($reportJob);
		
		return $reportJob;
	}
	
	function downloadReport($reportJob){
		$adManagerServices =  new AdManagerServices();
		$reportService = $adManagerServices->get(
			$this->session,
			ReportService::class
		);
		
		$reportDownloader = new ReportDownloader(
			$reportService,
			$reportJob->getId()
		);
		if ($reportDownloader->waitForReportToFinish()) {
			$options = new ReportDownloadOptions();
			$options->setExportFormat(ExportFormat::CSV_DUMP);
			$options->setUseGzipCompression(FALSE);
			
			$downloadUrl = $reportService->getReportDownloadUrlWithOptions($reportJob->getId(), $options);
			
			// Write to system temp directory by default.
			$filePath = sprintf(
				'%s.csv',
				tempnam(sys_get_temp_dir(), 'delivery-report-')
			);
			
			$ch = curl_init();
			curl_setopt($ch, CURLOPT_URL, $downloadUrl);
			curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
			$contents = curl_exec($ch);
			curl_close($ch);
			

			return $contents;
		}
		else {
			return false;
		}
	}
	
	function parseReport($content){
		$array = array_map('str_getcsv', explode(PHP_EOL,$content));
		if(empty($array)){
			return false;
		}
		
		$columns = array_shift($array);
		foreach($columns as $key=>$value){
			$columns[$key] = str_replace(['dimension.','dimensionattribute.','column.'], '', strtolower($value));
		}
		
		foreach($array as $key=>$value){
			if(empty($value) || count($value)==1){
				unset($array[$key]);
			}
		}
		
		foreach($array as $key=>$value){
			$array[$key] = array_combine($columns, $value);
		}
		
		return $array;
	}
	
	function getAdvertisers(){
		$this->connect();
		
		$advertisers = [];
		
		$serviceFactory = new ServiceFactory();
		$companyService = $serviceFactory->createCompanyService($this->session);
		// Create a statement to select companies.
		$pageSize = StatementBuilder::SUGGESTED_PAGE_LIMIT;
		$statementBuilder = (new StatementBuilder())->where('type = :type')
			->orderBy('id ASC')
			->limit($pageSize)
			->withBindVariableValue('type', CompanyType::ADVERTISER);
		// Retrieve a small amount of companies at a time, paging
		// through until all companies have been retrieved.
		$totalResultSetSize = 0;
		do {
			$page = $companyService->getCompaniesByStatement(
				$statementBuilder->toStatement()
			);
			// Print out some information for each company.
			if ($page->getResults() !== null) {
				$totalResultSetSize = $page->getTotalResultSetSize();
				$i = $page->getStartIndex();
				foreach ($page->getResults() as $company) {
					$advertisers[] = [
						'id' => $company->getId(),
						'name' => $company->getName(),
					];
				}
			}
			$statementBuilder->increaseOffsetBy($pageSize);
		} while ($statementBuilder->getOffset() < $totalResultSetSize);
		
		return $advertisers;
	}
	
	function getAdNetworks(){
		$this->connect();
		
		$advertisers = [];
		
		$serviceFactory = new ServiceFactory();
		$companyService = $serviceFactory->createCompanyService($this->session);
		// Create a statement to select companies.
		$pageSize = StatementBuilder::SUGGESTED_PAGE_LIMIT;
		$statementBuilder = (new StatementBuilder())->where('type = :type')
			->orderBy('id ASC')
			->limit($pageSize)
			->withBindVariableValue('type', CompanyType::AD_NETWORK);
		// Retrieve a small amount of companies at a time, paging
		// through until all companies have been retrieved.
		$totalResultSetSize = 0;
		do {
			$page = $companyService->getCompaniesByStatement(
				$statementBuilder->toStatement()
			);
			// Print out some information for each company.
			if ($page->getResults() !== null) {
				$totalResultSetSize = $page->getTotalResultSetSize();
				$i = $page->getStartIndex();
				foreach ($page->getResults() as $company) {
					$advertisers[] = [
						'id' => $company->getId(),
						'name' => $company->getName(),
					];
				}
			}
			$statementBuilder->increaseOffsetBy($pageSize);
		} while ($statementBuilder->getOffset() < $totalResultSetSize);
		
		return $advertisers;
	}
	
	function getAddUnits(){
		$this->connect();
		
		$ad_units = [];
		
		$serviceFactory = new ServiceFactory();
		$inventoryService = $serviceFactory->createInventoryService($this->session);
		// Create a statement to select ad units.
		$pageSize = StatementBuilder::SUGGESTED_PAGE_LIMIT;
		$statementBuilder = (new StatementBuilder())->where('status = :status')->withBindVariableValue(
				'status',
				'Active'
			)->orderBy('id ASC')
			->limit($pageSize);
		// Retrieve a small amount of ad units at a time, paging
		// through until all ad units have been retrieved.
		$totalResultSetSize = 0;
		do {
			$page = $inventoryService->getAdUnitsByStatement(
				$statementBuilder->toStatement()
			);
			// Print out some information for each ad unit.
			if ($page->getResults() !== null) {
				$totalResultSetSize = $page->getTotalResultSetSize();
				$i = $page->getStartIndex();
				foreach ($page->getResults() as $adUnit) {
					$ad_units[] = [
						'id' => $adUnit->getId(),
						'name' => $adUnit->getName()
					];
				}
			}
			$statementBuilder->increaseOffsetBy($pageSize);
		} while ($statementBuilder->getOffset() < $totalResultSetSize);
		
		return $ad_units;

	}
	
	function getOrders(){
		$this->connect();
		
		$orders = [];
		
		$serviceFactory = new ServiceFactory();
		$orderService = $serviceFactory->createOrderService($this->session);
		// Create a statement to select orders.
		$pageSize = StatementBuilder::SUGGESTED_PAGE_LIMIT;
		$statementBuilder = (new StatementBuilder())->orderBy('id ASC')
			->limit($pageSize);
		// Retrieve a small amount of orders at a time, paging
		// through until all orders have been retrieved.
		$totalResultSetSize = 0;
		do {
			$page = $orderService->getOrdersByStatement(
				$statementBuilder->toStatement()
			);
			// Print out some information for each order.
			if ($page->getResults() !== null) {
				$totalResultSetSize = $page->getTotalResultSetSize();
				$i = $page->getStartIndex();
				foreach ($page->getResults() as $order) {
					$data = [
						'id' => $order->getId(),
						'name' => $order->getName(),
						'start' => $order->getStartDateTime(),
						'end' => $order->getEndDateTime(),
						'archived' => $order->getIsArchived(),
						'advertiser_id' => $order->getAdvertiserId(),
						'status' => strtolower($order->getStatus())
					];
					
					if(!empty($data['start'])){
						$data['start'] = date('Y-m-d H:i:s',strtotime(AdManagerDateTimes::toDateTimeString($data['start'])));
					}
					if(!empty($data['end'])){
						$data['end'] = date('Y-m-d H:i:s',strtotime(AdManagerDateTimes::toDateTimeString($data['end'])));
					}
					
					$orders[] = $data;
				}
			}
			$statementBuilder->increaseOffsetBy($pageSize);
		} while ($statementBuilder->getOffset() < $totalResultSetSize);
		
		return $orders;
	}
	
	function createOrder($data){
		$whitelisted_fields = ['name','advertiser_id','startDateTime','endDateTime'];
		$this->connect();
		
		if(!empty($data['startDateTime'])){
			$data['startDateTime'] = AdManagerDateTimes::fromDateTime(
				new DateTime($data['startDateTime'], new DateTimeZone('America/Vancouver'))
			);
		}
		
		if(!empty($data['endDateTime'])){
			$data['endDateTime'] = AdManagerDateTimes::fromDateTime(
				new DateTime($data['endDateTime'], new DateTimeZone('America/Vancouver'))
			);
		}
		
		
		$serviceFactory = new ServiceFactory();
		$orderService = $serviceFactory->createOrderService($this->session);
		
		$order = new Order();
		$order->setTraffickerId($this->primary_trafficker_id);

		foreach($data as $key=>$value){
			if(in_array($key, $whitelisted_fields)){
				$func = 'set'.Inflector::camelize($key);
				$order->$func($value);
			}
		}

		$results = $orderService->createOrders([$order]);
		foreach ($results as $i => $order) {
			$id = $order->getId();
			$this->approveOrder($id);
			return $id;
		}
		
		return false;
	}
	
	function updateOrder($order_id, $data){
		$whitelisted_fields = ['name','advertiser_id','startDateTime','endDateTime'];
		$this->connect();
		
		$serviceFactory = new ServiceFactory();
		$orderService = $serviceFactory->createOrderService($this->session);
		
		$statementBuilder = new StatementBuilder();
		$statementBuilder->Where('id = :id');
		$statementBuilder->OrderBy('id ASC');
		$statementBuilder->Limit(1);
		$statementBuilder->WithBindVariableValue('id', $order_id);
		
		$page = $orderService->getOrdersByStatement(
			$statementBuilder->toStatement()
		);
		
		$results = $page->getResults();
		if(!empty($results)){
			$order = $results[0];
			
			foreach($data as $key=>$value){
				if(in_array($key, $whitelisted_fields)){
					$func = 'set'.Inflector::camelize($key);
					$order->$func($value);
				}
			}
			
			$orders = $orderService->updateOrders([$order]);
			return true;
		}
		
		return false;
	}
	
	function approveOrder($order_id){
		$this->connect();
		
		$serviceFactory = new ServiceFactory();
		$orderService = $serviceFactory->createOrderService($this->session);
		
		$statementBuilder = new StatementBuilder();
		$statementBuilder->Where('id = :id');
		$statementBuilder->OrderBy('id ASC');
		$statementBuilder->Limit(1);
		$statementBuilder->WithBindVariableValue('id', $order_id);
		
		$action = new ApproveOrdersAction();
		$result = $orderService->performOrderAction(
			$action,
			$statementBuilder->toStatement()
		);
		
		if ($result !== null && $result->getNumChanges() > 0) {
			return true;
		}
		
		return false;
	}
	
	function getLineItems(){
		$this->connect();
		
		$line_items = [];
		
		$serviceFactory = new ServiceFactory();
		$lineItemService = $serviceFactory->createLineItemService($this->session);
		// Create a statement to select line items.
		$pageSize = StatementBuilder::SUGGESTED_PAGE_LIMIT;
		$statementBuilder = (new StatementBuilder())->orderBy('id ASC')
			->limit($pageSize);
		// Retrieve a small amount of line items at a time, paging
		// through until all line items have been retrieved.
		$totalResultSetSize = 0;
		do {
			$page = $lineItemService->getLineItemsByStatement(
				$statementBuilder->toStatement()
			);
			// Print out some information for each line item.
			if ($page->getResults() !== null) {
				$totalResultSetSize = $page->getTotalResultSetSize();
				$i = $page->getStartIndex();
				foreach($page->getResults() as $lineItem){
					//$targeting = $lineItem->getTargeting();
					//$custom_targeting = $targeting->getCustomTargeting();
					$primaryGoal = $lineItem->getPrimaryGoal();
					$stats = $lineItem->getStats();

					$data = [
						'id' => $lineItem->getId(),
						'name' => $lineItem->getName(),
						'start' => $lineItem->getStartDateTime(),
						'end' => $lineItem->getEndDateTime(),
						'archived' => $lineItem->getIsArchived(),
						'order_id' => $lineItem->getOrderId(),
						'type' => strtolower($lineItem->getLineItemType()),
						'status' => strtolower($lineItem->getStatus()),
						'primary_goal_type' => '',
						'primary_goal_unit_type' => '',
						'primary_goal_units' => 0,
						'impressions_delivered' => 0,
						'clicks_delivered' => 0
					];
					
					if(!empty($primaryGoal)){
						$data['primary_goal_type'] = strtolower($primaryGoal->getGoalType());
						$data['primary_goal_unit_type'] = strtolower($primaryGoal->getUnitType());
						$data['primary_goal_units'] = $primaryGoal->getUnits();
					}
					
					if(!empty($stats)){
						$data['impressions_delivered'] = $stats->getImpressionsDelivered();
						$data['clicks_delivered'] = $stats->getClicksDelivered();
					}
					
					
					
					if(!empty($data['start'])){
						$data['start'] = date('Y-m-d H:i:s',strtotime(AdManagerDateTimes::toDateTimeString($data['start'])));
					}
					if(!empty($data['end'])){
						$data['end'] = date('Y-m-d H:i:s',strtotime(AdManagerDateTimes::toDateTimeString($data['end'])));
					}
					
					$line_items[] = $data;
				}
			}
			$statementBuilder->increaseOffsetBy($pageSize);
		} while ($statementBuilder->getOffset() < $totalResultSetSize);
		
		return $line_items;
	}
	
	function _customTargetingToArray($targeting){
		$custom_targeting = [];
	
		foreach($targeting as $target){
			//debug($target);
			if($target instanceof \Google\AdsApi\AdManager\v202011\CustomCriteriaSet){
				$temp = [
					'CustomCriteriaSet' => [
						'operator' => $target->getLogicalOperator(),
						'children' => []
					]
				];
				if(!empty($target->getChildren())){
					$temp['CustomCriteriaSet']['children'] = $this->_customTargetingToArray($target->getChildren());
				}
				$custom_targeting[] = $temp;
			}
			else if($target instanceof \Google\AdsApi\AdManager\v202011\CustomCriteria){
				$temp = [
					'CustomCriteria' => [
						'key_id' => $target->getKeyId(),
						'value_ids' => $target->getValueIds(),
						'operator' => $target->getOperator()
					]
				];
				$custom_targeting[] = $temp;
			}
		}
		
		if(count($custom_targeting) == 1 && isset($custom_targeting[0]['CustomCriteriaSet'])){
			$custom_targeting = $custom_targeting[0];
		}
		
		
		return $custom_targeting;
	}
	
	function getLineItem($line_item_id){
		$this->connect();
		
		$serviceFactory = new ServiceFactory();
		$lineItemService = $serviceFactory->createLineItemService($this->session);

		$statementBuilder = (new StatementBuilder())->where('id = :id')->orderBy('id ASC')
			->limit(1)->withBindVariableValue('id', $line_item_id);
		$totalResultSetSize = 0;
		do {
			$page = $lineItemService->getLineItemsByStatement(
				$statementBuilder->toStatement()
			);
			// Print out some information for each line item.
			if ($page->getResults() !== null) {
				$totalResultSetSize = $page->getTotalResultSetSize();
				$i = $page->getStartIndex();
				foreach($page->getResults() as $lineItem){
					$targeting = $lineItem->getTargeting();
					$customTargeting = $targeting->getCustomTargeting();
					$custom_targeting = $this->_customTargetingToArray([$customTargeting]);
					$primaryGoal = $lineItem->getPrimaryGoal();
					$stats = $lineItem->getStats();
					
					$data = [
						'id' => $lineItem->getId(),
						'name' => $lineItem->getName(),
						'start' => $lineItem->getStartDateTime(),
						'end' => $lineItem->getEndDateTime(),
						'archived' => $lineItem->getIsArchived(),
						'order_id' => $lineItem->getOrderId(),
						'type' => strtolower($lineItem->getLineItemType()),
						'status' => strtolower($lineItem->getStatus()),
						'targeting' => [
							'custom_targeting' => $custom_targeting
						],
						'primary_goal_type' => '',
						'primary_goal_unit_type' => '',
						'primary_goal_units' => 0,
						'impressions_delivered' => $stats->getImpressionsDelivered(),
						'clicks_delivered' => $stats->getClicksDelivered()
					];
					
					if(!empty($primaryGoal)){
						$data['primary_goal_type'] = strtolower($primaryGoal->getGoalType());
						$data['primary_goal_unit_type'] = strtolower($primaryGoal->getUnitType());
						$data['primary_goal_units'] = $primaryGoal->getUnits();
					}
					
					if(!empty($data['start'])){
						$data['start'] = date('Y-m-d H:i:s',strtotime(AdManagerDateTimes::toDateTimeString($data['start'])));
					}
					if(!empty($data['end'])){
						$data['end'] = date('Y-m-d H:i:s',strtotime(AdManagerDateTimes::toDateTimeString($data['end'])));
					}
					
					return $data;
				}
			}
			$statementBuilder->increaseOffsetBy($pageSize);
		} while ($statementBuilder->getOffset() < $totalResultSetSize);
		
		return false;
	}
	
	function updateLineItem($line_item_id, $data){
		$whitelisted_fields = ['name','startDateTime','endDateTime','startDateTimeType','unlimitedEndDateTime'];
		$this->connect();
		
		if(!empty($data['startDateTime'])){
			if($data['startDateTime']<date('Y-m-d H:i:s')){
				$data['startDateTimeType'] = StartDateTimeType::IMMEDIATELY;
				unset($data['startDateTime']);
			}
			else {
				$data['startDateTimeType'] = StartDateTimeType::USE_START_DATE_TIME;
				$data['startDateTime'] = AdManagerDateTimes::fromDateTime(
					new DateTime($data['startDateTime'], new DateTimeZone('America/Vancouver'))
				);
			}
		}
		
		if(!empty($data['endDateTime'])){
			$data['unlimitedEndDateTime'] = false;
			$data['endDateTime'] = AdManagerDateTimes::fromDateTime(
				new DateTime($data['endDateTime'], new DateTimeZone('America/Vancouver'))
			);
		}
		
		$serviceFactory = new ServiceFactory();
		$lineItemService = $serviceFactory->createLineItemService($this->session);
		
		$statementBuilder = new StatementBuilder();
		$statementBuilder->Where('id = :id');
		$statementBuilder->OrderBy('id ASC');
		$statementBuilder->Limit(1);
		$statementBuilder->WithBindVariableValue('id', $line_item_id);
		
		$page = $lineItemService->getLineItemsByStatement(
			$statementBuilder->toStatement()
		);
		
		$results = $page->getResults();
		if(!empty($results)){
			$line_item = $results[0];
			
			foreach($data as $key=>$value){
				if(in_array($key, $whitelisted_fields)){
					$func = 'set'.Inflector::camelize($key);
					$line_item->$func($value);
				}
			}
			
			$line_items = $lineItemService->updatelineItems([$line_item]);
			return true;
		}
		
		return false;
	}
	
	
	function getCreatives(){
		$this->connect();
		
		$creatives = [];
		
		$serviceFactory = new ServiceFactory();
		$creativeService = $serviceFactory->createCreativeService($this->session);
		// Create a statement to select creatives.
		$pageSize = StatementBuilder::SUGGESTED_PAGE_LIMIT;
		$statementBuilder = (new StatementBuilder())
			//->where('creativeType IN(:creativeType)')
			->orderBy('id ASC')
			->limit($pageSize)
			//->withBindVariableValue('creativeType', 'ImageCreative')
		;
		// Retrieve a small amount of creatives at a time, paging
		// through until all creatives have been retrieved.
		$totalResultSetSize = 0;
		do {
			$page = $creativeService->getCreativesByStatement(
				$statementBuilder->toStatement()
			);
			// Print out some information for each creative.
			if ($page->getResults() !== null) {
				$totalResultSetSize = $page->getTotalResultSetSize();
				$i = $page->getStartIndex();
				foreach ($page->getResults() as $creative) {

					$data = [
						'id' => $creative->getId(),
						'name' => $creative->getName(),
						'advertiser_id' => $creative->getAdvertiserId(),
						'destination_url' => '',
						'width' => $creative->getSize()->getWidth(),
						'height' => $creative->getSize()->getHeight(),
						'preview_url' => $creative->getPreviewUrl()
					];
					
					if($creative instanceof \Google\AdsApi\AdManager\v202011\TemplateCreative){
						$templateVariables = $creative->getCreativeTemplateVariableValues();
						foreach($templateVariables as $variable){
							if($variable->getUniqueName() == 'ClickThroughURL'){
								if(!empty($variable->getValue())){
									$data['destination_url'] = $variable->getValue();
								}
							}
						}

					}
					else if($creative instanceof \Google\AdsApi\AdManager\v202011\ImageCreative){
						$data['destination_url'] = $creative->getDestinationUrl();
					}
					else if($creative instanceof \Google\AdsApi\AdManager\v202011\InternalRedirectCreative){
						$data['destination_url'] = $creative->getInternalRedirectUrl();
					}
					else if($creative instanceof \Google\AdsApi\AdManager\v202011\Html5Creative){
						$click_tags = $creative->getHtml5Asset()->getClickTags();
						if(!empty($click_tags)){
							$data['destination_url'] = $click_tags[0]->getUrl();
						}
					}
					
					$creatives[] = $data;

				}
			}
			$statementBuilder->increaseOffsetBy($pageSize);
		} while ($statementBuilder->getOffset() < $totalResultSetSize);


		return $creatives;
	}
	
	function getCreative($creative_id){
		$this->connect();
		
		$creatives = [];
		
		$serviceFactory = new ServiceFactory();
		$creativeService = $serviceFactory->createCreativeService($this->session);
		$pageSize = 1;
		$statementBuilder = (new StatementBuilder())
			->where('id = :creativeId')
			->orderBy('id ASC')
			->limit($pageSize)
			->withBindVariableValue('creativeId', $creative_id)
		;
		// Retrieve a small amount of creatives at a time, paging
		// through until all creatives have been retrieved.
		$totalResultSetSize = 0;
		do {
			$page = $creativeService->getCreativesByStatement(
				$statementBuilder->toStatement()
			);
			// Print out some information for each creative.
			if ($page->getResults() !== null) {
				$totalResultSetSize = $page->getTotalResultSetSize();
				$i = $page->getStartIndex();
				foreach ($page->getResults() as $creative) {
					$data = [
						'id' => $creative->getId(),
						'name' => $creative->getName(),
						'advertiser_id' => $creative->getAdvertiserId(),
						'destination_url' => '',
						'width' => $creative->getSize()->getWidth(),
						'height' => $creative->getSize()->getHeight(),
						'preview_url' => $creative->getPreviewUrl()
					];
					
					if($creative instanceof \Google\AdsApi\AdManager\v202011\TemplateCreative){
						$templateVariables = $creative->getCreativeTemplateVariableValues();
						foreach($templateVariables as $variable){
							if($variable->getUniqueName() == 'ClickThroughURL'){
								if(!empty($variable->getValue())){
									$data['destination_url'] = $variable->getValue();
								}
							}
						}

					}
					else if($creative instanceof \Google\AdsApi\AdManager\v202011\ImageCreative){
						$data['destination_url'] = $creative->getDestinationUrl();
					}
					else if($creative instanceof \Google\AdsApi\AdManager\v202011\InternalRedirectCreative){
						$data['destination_url'] = $creative->getInternalRedirectUrl();
					}
					else if($creative instanceof \Google\AdsApi\AdManager\v202011\Html5Creative){
						$click_tags = $creative->getHtml5Asset()->getClickTags();
						if(!empty($click_tags)){
							$data['destination_url'] = $click_tags[0]->getUrl();
						}
					}
					
					return $data;

				}
			}
			$statementBuilder->increaseOffsetBy($pageSize);
		} while ($statementBuilder->getOffset() < $totalResultSetSize);


		return false;
	}
	
	function getCreativeAssociations(){
		$this->connect();
		
		$associations = [];
		
		$serviceFactory = new ServiceFactory();
		$lineItemCreativeAssociationService = $serviceFactory->createLineItemCreativeAssociationService($this->session);
		// Create a statement to select line item creative associations.
		$pageSize = StatementBuilder::SUGGESTED_PAGE_LIMIT;
		$statementBuilder = (new StatementBuilder())
			->orderBy('lineItemId ASC, creativeId ASC')
			->limit($pageSize);
		// Retrieve a small amount of line item creative associations at a time,
		// paging through until all line item creative associations have been
		// retrieved.
		$totalResultSetSize = 0;
		do {
			$page = $lineItemCreativeAssociationService
				->getLineItemCreativeAssociationsByStatement(
					$statementBuilder->toStatement()
				);
			// Print out some information for each line item creative
			// association.
			if ($page->getResults() !== null) {
				$totalResultSetSize = $page->getTotalResultSetSize();
				$i = $page->getStartIndex();
				foreach ($page->getResults() as $lica) {

					if ($lica->getCreativeSetId() !== null) {
						/*
						$lica->getLineItemId(),
						$lica->getCreativeSetId(),
						*/
					} else {
						$data = [
							'line_item_id' => $lica->getLineItemId(),
							'creative_id' => $lica->getCreativeId(),
							'destination_url' => '',
							'start' => $lica->getStartDateTime(),
							'end' => $lica->getEndDateTime(),
							'start_date_type' => strtolower($lica->getStartDateTimeType()),
							'status' => strtolower($lica->getStatus())
						];
						
						$destination_url = $lica->getDestinationUrl();
						
						if(!empty($destination_url)){
							$data['destination_url'] = $destination_url;
						}
						
						if(!empty($data['start'])){
							$data['start'] = date('Y-m-d H:i:s',strtotime(AdManagerDateTimes::toDateTimeString($data['start'])));
						}
						if(!empty($data['end'])){
							$data['end'] = date('Y-m-d H:i:s',strtotime(AdManagerDateTimes::toDateTimeString($data['end'])));
						}
						
						$associations[] = $data;
					}
					
				}
			}
			$statementBuilder->increaseOffsetBy($pageSize);
		} while ($statementBuilder->getOffset() < $totalResultSetSize);
		
		return $associations;
	}
	
	function getCreativePreviewUrl($line_item_id, $creative_id, $url){
		$this->connect();
		
		$serviceFactory = new ServiceFactory();
		$pageSize = 1;
		$lineItemCreativeAssociationService = $serviceFactory->createLineItemCreativeAssociationService($this->session);
		
		$preview_url = $lineItemCreativeAssociationService->getPreviewUrl($line_item_id, $creative_id, $url);
		if(!empty($preview_url)){
			return $preview_url;
		}
		
		return false;
	}
	
	function updateCompany($company_id, $data){
		$whitelisted_fields = ['name','type','address','email','primaryPhone'];
		$this->connect();
		
		$serviceFactory = new ServiceFactory();
		$companyService = $serviceFactory->createCompanyService($this->session);
		// Create a statement to select companies.
		$statementBuilder = (new StatementBuilder())->where('id = :id')
			->limit(1)
			->withBindVariableValue('id', $company_id);


		$page = $companyService->getCompaniesByStatement(
			$statementBuilder->toStatement()
		);

		if ($page->getResults() !== null) {
			$company = $page->getResults();

			if(!empty($company)){
				$company = $company[0];
			}
			
			foreach($data as $key=>$value){
				if(in_array($key, $whitelisted_fields)){
					$func = 'set'.Inflector::camelize($key);
					$company->$func($value);
				}
			}

			$results = $companyService->updateCompanies([$company]);
			
			return true;
		}

		return false;
	}
	
	function createCompany($data){
		$whitelisted_fields = ['name','type','address','email','primaryPhone'];
		$this->connect();
		
		$serviceFactory = new ServiceFactory();
		$companyService = $serviceFactory->createCompanyService($this->session);
		
		$company = new Company();
		$company->setType(CompanyType::ADVERTISER);
		
		foreach($data as $key=>$value){
			if(in_array($key, $whitelisted_fields)){
				$func = 'set'.Inflector::camelize($key);
				$company->$func($value);
			}
		}
		
		$results = $companyService->createCompanies([$company]);
		foreach ($results as $company) {
			return $company->getId();
		}

		return false;
	}
	
	function getCustomTargetingValues($predefinedOnly=false){
		$this->connect();
		
		$targeting = [];
		
		$serviceFactory = new ServiceFactory();
		$customTargetingService = $serviceFactory->createCustomTargetingService($this->session);
		
		$customTargetingKeys = $this->getCustomTargetingKeys($serviceFactory, $this->session);
		
		
		$pageSize = StatementBuilder::SUGGESTED_PAGE_LIMIT;

		$statementBuilder =
			(new StatementBuilder())
				->where('customTargetingKeyId = :customTargetingKeyId')
				->orderBy('id ASC')
				->limit($pageSize);
		
		$values = [];
		
		// For each key, retrieve all its values.
		$totalValueCounter = 0;
		foreach ($customTargetingKeys as $key) {
			if($predefinedOnly && $key['type'] != 'predefined'){
				continue;
			}
			// Set the custom targeting key ID to select from.
			$statementBuilder->withBindVariableValue(
				'customTargetingKeyId',
				$key['id']
			);
			// Retrieve a small amount of custom targeting values at a time,
			// paging through until all custom targeting values have been
			// retrieved.
			$totalResultSetSize = 0;
			$statementBuilder->offset(0);
			do {
				$page = $customTargetingService
					->getCustomTargetingValuesByStatement(
						$statementBuilder->toStatement()
					);
				// Print out some information for each custom targeting value.
				if ($page->getResults() !== null) {
					$totalResultSetSize = $page->getTotalResultSetSize();
					foreach ($page->getResults() as $customTargetingValue) {
						$values[] = [
							'id' => $customTargetingValue->getId(),
							'name' => $customTargetingValue->getName(),
							'display_name' => $customTargetingValue->getDisplayName(),
							'match_type' => strtolower($customTargetingValue->getMatchType()),
							'status' => strtolower($customTargetingValue->getStatus()),
							'key_id' => $customTargetingValue->getCustomTargetingKeyId()
						];
					}
				}
				$statementBuilder->increaseOffsetBy($pageSize);
			} while ($statementBuilder->getOffset() < $totalResultSetSize);
		}

		return $values;
	}
	
	function getCustomTargetingKeys(){
		$this->connect();
		
		$keys = [];
		
		$serviceFactory = new ServiceFactory();
		$customTargetingService = $serviceFactory->createCustomTargetingService($this->session);
		
		$pageSize = StatementBuilder::SUGGESTED_PAGE_LIMIT;
		$statementBuilder = (new StatementBuilder())->orderBy('id ASC')
			->limit($pageSize);
		// Retrieve a small amount of custom targeting keys at a time, paging
		// through until all custom targeting keys have been retrieved.
		$totalResultSetSize = 0;
		do {
			$page = $customTargetingService->getCustomTargetingKeysByStatement(
				$statementBuilder->toStatement()
			);
			if ($page->getResults() !== null) {
				$totalResultSetSize = $page->getTotalResultSetSize();
				$i = $page->getStartIndex();
				foreach ($page->getResults() as $customTargetingKey) {
					$keys[] = [
						'id' => $customTargetingKey->getId(),
						'name' => $customTargetingKey->getName(),
						'display_name' => $customTargetingKey->getDisplayName(),
						'type' => strtolower($customTargetingKey->getType()),
						'status' => strtolower($customTargetingKey->getStatus())
					];
				}
			}
			$statementBuilder->increaseOffsetBy($pageSize);
		} while ($statementBuilder->getOffset() < $totalResultSetSize);
		
		
		return $keys;
		
	}
	
}